<?php
/*
Plugin Name: SPC-BOJO
Description: A plugin to apply custom popup and discounts based on WooCommerce cart content. Includes category management, popup notifications, and discount handling.
Version: 1.0
Author: SPC
*/

if (!defined('ABSPATH')) {
    exit; // Prevent direct access
}

// פונקציה מספר 1 - אבטחת הקלטים וכיבוי XML-RPC
function spc_bojo_secure() {
    // Disable XML-RPC
    add_filter('xmlrpc_enabled', '__return_false');

    // Protect against SQL Injection, XSS, and Spoofing by sanitizing inputs
    if (!empty($_POST)) {
        foreach ($_POST as $key => $value) {
            $_POST[$key] = sanitize_text_field($value);
        }
    }
}
add_action('init', 'spc_bojo_secure');

// פונקציה מספר 2 - טעינת סקריפטים וסגנונות
function spc_bojo_enqueue_assets() {
    wp_enqueue_style('spc-bojo-style', plugin_dir_url(__FILE__) . 'assets/css/spc-bojo-style.css');
    wp_enqueue_script('spc-bojo-scripts', plugin_dir_url(__FILE__) . 'assets/js/spc-bojo-scripts.js', array('jquery'), null, true);
}
add_action('wp_enqueue_scripts', 'spc_bojo_enqueue_assets');


// הוספת עמוד ניהול "הגדרת BOJO" לתפריט הצדדי של WordPress
function add_bojo_menu_page() {
    add_menu_page(
        'הגדרת BOJO',
        'הגדרת BOJO',
        'manage_options',
        'bojo-settings',
        'render_bojo_settings_page',
        'dashicons-admin-tools',
        20
    );
}
add_action('admin_menu', 'add_bojo_menu_page');

// פונקציית שמירת הנתונים
function save_bojo_settings() {
    if (isset($_POST['bojo_save_settings'])) {
        update_option('bojo_categories_1', isset($_POST['category_select_1']) ? implode(',', $_POST['category_select_1']) : '');
        update_option('bojo_required_quantity_1', sanitize_text_field($_POST['required_quantity_1']));
        update_option('bojo_free_quantity_1', sanitize_text_field($_POST['free_quantity_1']));
        update_option('bojo_total_discount_1', sanitize_text_field($_POST['total_discount_1']));
        update_option('bojo_h3_title_1', sanitize_text_field($_POST['bojo_h3_title_1']));
        update_option('bojo_paragraph_text_1', sanitize_text_field($_POST['bojo_paragraph_text_1']));

        // שמירת השדות עבור פופאפ 2
        update_option('bojo_h3_title_2', sanitize_text_field($_POST['bojo_h3_title_2']));
        update_option('bojo_paragraph_text_2', sanitize_text_field($_POST['bojo_paragraph_text_2']));

        // שמירת הלינקים המותאמים
        $custom_links = [];
        if (!empty($_POST['custom_links'])) {
            foreach ($_POST['custom_links'] as $category_id => $link) {
                $custom_links[$category_id] = esc_url_raw($link);
            }
        }
        update_option('bojo_custom_links', $custom_links);
    }
}
add_action('admin_init', 'save_bojo_settings');

// הפונקציה שתציג את תוכן עמוד הניהול
function render_bojo_settings_page() {
    $saved_categories = get_option('bojo_categories_1');
    $custom_links = get_option('bojo_custom_links', []); // קבלת הלינקים המותאמים
    $required_quantity = get_option('bojo_required_quantity_1');
    $free_quantity = get_option('bojo_free_quantity_1');
    $total_discount = get_option('bojo_total_discount_1');
    $bojo_h3_title_1 = get_option('bojo_h3_title_1');
    $bojo_paragraph_text_1 = get_option('bojo_paragraph_text_1');
    
    // הצגת השדות עבור פופאפ 2
    $bojo_h3_title_2 = get_option('bojo_h3_title_2');
    $bojo_paragraph_text_2 = get_option('bojo_paragraph_text_2');

    echo '<div class="wrap"><h1>הגדרת BOJO</h1>';
    echo '<form method="POST">';
    wp_nonce_field('bojo_save_settings'); // הגנה על טפסים

    // קטגוריות 1
    echo '
    <div class="bojo-category-box">
        <h2>הגדרות קטגוריות</h2>
        <div class="bojo-input-row">
            <div class="bojo-input-group">
                <label for="category_select_1">קטגוריות 1</label>
                <select name="category_select_1[]" id="category_select_1" class="wc-enhanced-select" multiple="multiple" data-placeholder="בחר קטגוריות...">';
                $categories = get_terms(array(
                    'taxonomy' => 'product_cat',
                    'hide_empty' => false,
                ));
                foreach ($categories as $category) {
                    $selected = in_array($category->term_id, explode(',', $saved_categories)) ? ' selected="selected"' : '';
                    echo '<option value="' . esc_attr($category->term_id) . '"' . $selected . '>' . esc_html($category->name) . '</option>';
                }
                echo '</select>
            </div>

            <div class="bojo-input-group">
                <label for="required_quantity_1">כמות נדרשת</label>
                <input type="number" min="0" name="required_quantity_1" id="required_quantity_1" value="' . esc_attr($required_quantity ? $required_quantity : 0) . '" />
            </div>

            <div class="bojo-input-group">
                <label for="free_quantity_1">כמות במתנה</label>
                <input type="number" min="0" name="free_quantity_1" id="free_quantity_1" value="' . esc_attr($free_quantity ? $free_quantity : 0) . '" />
            </div>

            <div class="bojo-input-group">
                <label for="total_discount_1">סך הכל להפעלת ההנחה</label>
                <input type="number" min="0" name="total_discount_1" id="total_discount_1" value="' . esc_attr($total_discount ? $total_discount : 0) . '" readonly />
            </div>
        </div>
    </div>';

    // שדות של פופאפ 1
    echo '
    <h2>הגדרות פופאפ 1</h2>
    <div class="bojo-input-group">
        <label for="bojo_h3_title_1">כותרת H3 פופאפ 1</label>
        <input type="text" name="bojo_h3_title_1" id="bojo_h3_title_1" value="' . esc_attr($bojo_h3_title_1 ? $bojo_h3_title_1 : '') . '" />
    </div>

    <div class="bojo-input-group">
        <label for="bojo_paragraph_text_1">תיאור פסקה פופאפ 1</label>
        <textarea name="bojo_paragraph_text_1" id="bojo_paragraph_text_1" rows="5">' . esc_textarea($bojo_paragraph_text_1 ? $bojo_paragraph_text_1 : '') . '</textarea>
    </div>';

    // שדות של פופאפ 2
    echo '
    <h2>הגדרות פופאפ 2</h2>
    <div class="bojo-input-group">
        <label for="bojo_h3_title_2">כותרת H3 פופאפ 2</label>
        <input type="text" name="bojo_h3_title_2" id="bojo_h3_title_2" value="' . esc_attr($bojo_h3_title_2 ? $bojo_h3_title_2 : '') . '" />
    </div>

    <div class="bojo-input-group">
        <label for="bojo_paragraph_text_2">תיאור פסקה פופאפ 2</label>
        <textarea name="bojo_paragraph_text_2" id="bojo_paragraph_text_2" rows="5">' . esc_textarea($bojo_paragraph_text_2 ? $bojo_paragraph_text_2 : '') . '</textarea>
    </div>';

    // הזנת לינקים מותאמים לכל קטגוריה
    if (!empty($saved_categories)) {
        echo '<h2>הזנת לינקים מותאמים לכל קטגוריה</h2>';
        $category_ids = explode(',', $saved_categories);
        foreach ($category_ids as $index => $category_id) {
            $category = get_term($category_id, 'product_cat');
            $custom_link = isset($custom_links[$category_id]) ? $custom_links[$category_id] : '';
            echo '
            <div class="bojo-input-group">
                <label for="custom_link_' . esc_attr($category_id) . '">לינק מותאם לקטגוריה ' . esc_html($category->name) . '</label>
                <input type="url" name="custom_links[' . esc_attr($category_id) . ']" id="custom_link_' . esc_attr($category_id) . '" value="' . esc_url($custom_link) . '" placeholder="הזן לינק מותאם" />
            </div>';
        }
    }

    echo '<div class="bojo-input-row"><button type="submit" name="bojo_save_settings" class="button button-primary">שמור</button></div>';
    echo '</form></div>';
}

// הוספת סקריפטים וסגנונות לממשק הניהול להשלמה אוטומטית
function enqueue_wc_select2() {
    if (is_admin()) {
        // טוען את סקריפט ה-autocomplete של WooCommerce (Select2)
        wp_enqueue_script('wc-enhanced-select');
        wp_enqueue_style('woocommerce_admin_styles'); // סגנון לממשק הניהול של WooCommerce
    }
}
add_action('admin_enqueue_scripts', 'enqueue_wc_select2');

// יצירת פופאפ 1 להצגת טקסט מהשדות או מהפלייסהולדר והוספת קטגוריות
function bojo_popup_content_1() {
    $h3_title_1 = get_option('bojo_h3_title_1', 'זה מגיע לכן מתנה');
    $paragraph_text_1 = get_option('bojo_paragraph_text_1', 'לרכישה');
    $saved_categories = get_option('bojo_categories_1');
    $custom_links = get_option('bojo_custom_links', []);

    echo '
    <div id="bojo-popup-1" class="bojo-popup">
        <div class="bojo-popup-content">
            <span class="bojo-popup-close">&times;</span>
            <h3>' . esc_html($h3_title_1) . '</h3>
            <p>' . esc_html($paragraph_text_1) . '</p>';

    if (!empty($saved_categories)) {
        $category_ids = explode(',', $saved_categories);
        echo '<div class="bojo-category-container">';
        foreach ($category_ids as $category_id) {
            $category = get_term($category_id, 'product_cat');
            $thumbnail_id = get_term_meta($category_id, 'thumbnail_id', true);
            $image_url = wp_get_attachment_url($thumbnail_id);
            $custom_link = isset($custom_links[$category_id]) ? $custom_links[$category_id] : '#';

            if ($category) {
                echo '<div class="bojo-category">';
                if ($image_url) {
                    echo '<a href="' . esc_url($custom_link) . '" target="_blank"><img src="' . esc_url($image_url) . '" alt="' . esc_attr($category->name) . '" /></a>';
                }
                echo '<h4><a href="' . esc_url($custom_link) . '" target="_blank">' . esc_html($category->name) . '</a></h4>';
                echo '</div>';
            }
        }
        echo '</div>';
    }

    echo '</div></div>';
}
add_action('wp_footer', 'bojo_popup_content_1');

// יצירת פופאפ 2 להצגת טקסט מהשדות או מהפלייסהולדר והוספת קטגוריות
function bojo_popup_content_2() {
    $h3_title_2 = get_option('bojo_h3_title_2', 'זה מגיע לכן מתנה');
    $paragraph_text_2 = get_option('bojo_paragraph_text_2', 'לרכישה');
    $saved_categories = get_option('bojo_categories_1');
    $custom_links = get_option('bojo_custom_links', []);

    echo '
    <div id="bojo-popup-2" class="bojo-popup">
        <div class="bojo-popup-content">
            <span class="bojo-popup-close">&times;</span>
            <h3>' . esc_html($h3_title_2) . '</h3>
            <p>' . esc_html($paragraph_text_2) . '</p>';

    if (!empty($saved_categories)) {
        $category_ids = explode(',', $saved_categories);
        echo '<div class="bojo-category-container">';
        foreach ($category_ids as $category_id) {
            $category = get_term($category_id, 'product_cat');
            $thumbnail_id = get_term_meta($category_id, 'thumbnail_id', true);
            $image_url = wp_get_attachment_url($thumbnail_id);
            $custom_link = isset($custom_links[$category_id]) ? $custom_links[$category_id] : '#';

            if ($category) {
                echo '<div class="bojo-category">';
                if ($image_url) {
                    echo '<a href="' . esc_url($custom_link) . '" target="_blank"><img src="' . esc_url($image_url) . '" alt="' . esc_attr($category->name) . '" /></a>';
                }
                echo '<h4><a href="' . esc_url($custom_link) . '" target="_blank">' . esc_html($category->name) . '</a></h4>';
                echo '</div>';
            }
        }
        echo '</div>';
    }

    echo '</div></div>';
}
add_action('wp_footer', 'bojo_popup_content_2');

// JavaScript להצגת הפופאפים כאשר מתקיימים התנאים בעגלת הקניות
add_action('wp_footer', function() {
    echo '
    <script type="text/javascript">
    document.addEventListener("DOMContentLoaded", function() {
        var popup1 = document.getElementById("bojo-popup-1");
        var popup2 = document.getElementById("bojo-popup-2");
        var closeBtns = document.querySelectorAll(".bojo-popup-close");

        // סגירת הפופאפים כאשר לוחצים על ה-X
        closeBtns.forEach(function(btn) {
            btn.onclick = function() {
                popup1.style.display = "none";
                popup2.style.display = "none";
            };
        });

        // סגירת הפופאפים כאשר לוחצים מחוץ לפופאפ
        window.onclick = function(event) {
            if (event.target == popup1) {
                popup1.style.display = "none";
            }
            if (event.target == popup2) {
                popup2.style.display = "none";
            }
        };

        // בדיקה אם יש 5 או 10 מוצרים בעגלה כדי להציג את הפופאפים
        function checkCart() {
            var cartQty = ' . WC()->cart->get_cart_contents_count() . ';

            if (cartQty == 5) {
                popup1.style.display = "block";
            }
            if (cartQty == 10) {
                popup2.style.display = "block";
            }
        }

        // קריאה לבדוק את העגלה מיד עם טעינת הדף
        checkCart();
    });
    </script>';
});

// עיצוב הפופאפים
add_action('wp_head', function() {
    echo '
    <style type="text/css">
    .bojo-popup {
        display: none; /* מוסתר בהתחלה */
        position: fixed;
        z-index: 9999; /* פתיחה מעל כל האלמנטים */
        left: 0;
        top: 0;
        width: 100%;
        height: 100%;
        overflow-y: auto; /* מאפשר גלילה אם התוכן גבוה מדי */
        background-color: rgba(0, 0, 0, 0.5); /* רקע חצי שקוף */
    }

    .bojo-popup-content {
        background-color: #fff;
        margin: 5% auto;
        padding: 20px;
        border: 1px solid #888;
        width: 80%;
        max-width: 700px;
        text-align: center;
        font-family: inherit;
        max-height: 90%; /* מגביל את הגובה של הפופאפ */
        overflow-y: auto; /* מאפשר גלילה בתוך הפופאפ */
    }

    .bojo-popup-close {
        color: #aaa;
        float: right;
        font-size: 28px;
        font-weight: bold;
        cursor: pointer;
    }

    .bojo-popup-close:hover,
    .bojo-popup-close:focus {
        color: black;
        text-decoration: none;
        cursor: pointer;
    }

    .bojo-popup-content h3 {
        font-size: 24px;
        margin-bottom: 10px;
        color: inherit; /* ירש מהתבנית */
    }

    .bojo-popup-content p {
        font-size: 18px;
        color: inherit; /* ירש מהתבנית */
    }

    .bojo-category-container {
        display: flex;
        flex-wrap: nowrap;
        justify-content: center;
        gap: 20px;
        margin-top: 20px;
    }

    .bojo-category {
        text-align: center;
        width: 150px;
    }

    .bojo-category img {
        max-width: 100px;
        height: auto;
        margin-bottom: 10px;
    }

    .bojo-category h4 {
        font-size: 16px;
        color: inherit;
    }
    </style>';
});


// החלת הנחה על כמות מוצרים במתנה מתוך המוצרים בעגלה
function bojo_apply_discount($cart) {
    if ( is_admin() && ! defined( 'DOING_AJAX' ) )
        return;

    $categories_to_check = get_option('bojo_categories_1'); // קטגוריות לבדיקה
    $free_quantity_6 = 1; // הנחה של מוצר אחד אם יש בדיוק 6 מוצרים
    $free_quantity_13 = 3; // הנחה של 3 מוצרים אם יש בדיוק 13 מוצרים
    $required_quantity_6 = 6; // הכמות הנדרשת להנחה אחת
    $required_quantity_13 = 13; // הכמות הנדרשת להנחה השנייה
    $total_discount_6 = 6; // סה"כ להפעלת ההנחה ל-6 מוצרים
    $total_discount_13 = 13; // סה"כ להפעלת ההנחה ל-13 מוצרים

    if ($categories_to_check) {
        $categories = explode(',', $categories_to_check);
        $eligible_items = [];
        $total_qty = 0;

        // מציאת מוצרים בקטגוריות הנבחרות והוספתם לרשימה
        foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
            $product_id = $cart_item['product_id'];
            $terms = get_the_terms($product_id, 'product_cat');
            $product_price = $cart_item['data']->get_price();

            if ($terms) {
                foreach ($terms as $term) {
                    if (in_array($term->term_id, $categories)) {
                        $total_qty += $cart_item['quantity'];
                        // הוספת המוצר לרשימת המוצרים הזכאים
                        $eligible_items[] = [
                            'key' => $cart_item_key,
                            'price' => $product_price,
                            'quantity' => $cart_item['quantity'],
                            'data' => $cart_item['data']
                        ];
                    }
                }
            }
        }

        // תנאי ראשון: אם יש בדיוק 6 מוצרים, הענק הנחה על מוצר אחד
        if ($total_qty == $total_discount_6) {
            // מיון מוצרים לפי המחיר (הזולים ביותר תחילה)
            usort($eligible_items, function($a, $b) {
                return $a['price'] - $b['price'];
            });

            $remaining_free_qty = $free_quantity_6;

            // החלת ההנחה על המוצרים הזולים ביותר
            foreach ($eligible_items as $item) {
                if ($remaining_free_qty <= 0) {
                    break;
                }

                $apply_qty = min($item['quantity'], $remaining_free_qty);

                // חלוקת ההנחה אם יש מספר יחידות מאותו מוצר
                if ($apply_qty < $item['quantity']) {
                    // חישוב אחוז הנחה עבור כל יחידה
                    $discount_percentage = $apply_qty / $item['quantity'];

                    // חישוב המחיר לאחר ההנחה
                    $discounted_price = $item['price'] * (1 - $discount_percentage);

                    // עדכון המחיר לאחר החלת ההנחה
                    $cart_item = $cart->get_cart_item($item['key']);
                    $cart_item['data']->set_price($discounted_price);

                    $remaining_free_qty -= $apply_qty; // מפחיתים את כמות המוצרים שקיבלו מתנה
                } else {
                    // אם כל היחידות של המוצר מקבלות הנחה מלאה
                    $cart_item = $cart->get_cart_item($item['key']);
                    $cart_item['data']->set_price(0); // הנחה של 100% למוצר אחד במתנה
                    $remaining_free_qty -= $item['quantity']; // מפחיתים את הכמות שניתן להנחה
                }
            }
        }

        // תנאי שני: אם יש בדיוק 13 מוצרים, הענק הנחה על 3 מוצרים לפי התנאים
        elseif ($total_qty == $total_discount_13) {
            $remaining_free_qty = $free_quantity_13;
            $single_unit_items = [];
            $two_unit_items = [];
            $three_unit_items = [];

            // מציאת מוצרים עם יחידה אחת, שתי יחידות או שלוש יחידות
            foreach ($eligible_items as $item) {
                if ($item['quantity'] == 1) {
                    $single_unit_items[] = $item;
                } elseif ($item['quantity'] == 2) {
                    $two_unit_items[] = $item;
                } elseif ($item['quantity'] == 3) {
                    $three_unit_items[] = $item;
                }
            }

            // אם יש מוצר עם בדיוק 3 יחידות, תוריד אותו קודם
            if (count($three_unit_items) > 0) {
                // מיון מוצרים לפי המחיר (הזולים ביותר תחילה)
                usort($three_unit_items, function($a, $b) {
                    return $a['price'] - $b['price'];
                });

                // הנחה מלאה על מוצר עם 3 יחידות
                $item = $three_unit_items[0]; // המוצר הזול ביותר שיש לו בדיוק 3 יחידות
                $cart_item = $cart->get_cart_item($item['key']);
                $cart_item['data']->set_price(0); // הנחה של 100% למוצר עם 3 יחידות
                $remaining_free_qty -= 3;
            }

            // אם יש לפחות 3 מוצרים עם יחידה אחת, תן הנחה ל-3 מהם
            elseif (count($single_unit_items) >= 3) {
                // מיון מוצרים לפי המחיר (הזולים ביותר תחילה)
                usort($single_unit_items, function($a, $b) {
                    return $a['price'] - $b['price'];
                });

                foreach (array_slice($single_unit_items, 0, 3) as $item) {
                    $cart_item = $cart->get_cart_item($item['key']);
                    $cart_item['data']->set_price(0); // הנחה של 100% ליחידה אחת
                    $remaining_free_qty--; // הפחתת כמות הפריטים במתנה שנותרו
                }
            } 
            // אם אין 3 מוצרים שונים עם יחידה אחת, מצא מוצר עם 2 יחידות ומוצר אחד עם יחידה אחת
            elseif (count($two_unit_items) > 0 && count($single_unit_items) > 0) {
                // מיון מוצרים לפי המחיר (הזולים ביותר תחילה)
                usort($two_unit_items, function($a, $b) {
                    return $a['price'] - $b['price'];
                });

                // הנחה מלאה על 2 יחידות
                $item = $two_unit_items[0];
                $cart_item = $cart->get_cart_item($item['key']);
                $cart_item['data']->set_price(0); // הנחה של 100% על 2 היחידות
                $remaining_free_qty -= 2;

                // הענקת הנחה על מוצר עם יחידה אחת אם קיים
                if (count($single_unit_items) > 0) {
                    usort($single_unit_items, function($a, $b) {
                        return $a['price'] - $b['price'];
                    });

                    $item = $single_unit_items[0];
                    $cart_item = $cart->get_cart_item($item['key']);
                    $cart_item['data']->set_price(0); // הנחה של 100% ליחידה אחת
                    $remaining_free_qty--;
                }
            } 
            // אם לא נמצאו 3 יחידות מאף סוג אחר, מצא מוצר עם 4 עד 12 יחידות
            else {
                $four_to_twelve_unit_items = [];

                foreach ($eligible_items as $item) {
                    if ($item['quantity'] >= 4 && $item['quantity'] <= 12) {
                        $four_to_twelve_unit_items[] = $item;
                    }
                }

                // מצא את המוצר עם הכמות הנמוכה ביותר בין 4 ל-12 עם עדיפות למספר שמתחלק ב-3
                usort($four_to_twelve_unit_items, function($a, $b) {
                    // עדיפות למספרים שמתחלקים ב-3
                    if ($a['quantity'] % 3 == 0 && $b['quantity'] % 3 != 0) {
                        return -1;
                    } elseif ($a['quantity'] % 3 != 0 && $b['quantity'] % 3 == 0) {
                        return 1;
                    } else {
                        // אם אף אחד מהם לא מתחלק ב-3, נבחר את הקטן יותר
                        return $a['quantity'] - $b['quantity'];
                    }
                });

                if (count($four_to_twelve_unit_items) > 0) {
                    $item = $four_to_twelve_unit_items[0]; // המוצר המתאים ביותר
                    $cart_item = $cart->get_cart_item($item['key']);

                    // חישוב השבר להנחה (3 חלקי הכמות)
                    $discount_percentage = 3 / $item['quantity'];

                    // הנחה חלקית לפי השבר
                    $discounted_price = $item['price'] * (1 - $discount_percentage);
                    $cart_item['data']->set_price($discounted_price); // הנחה לפי השבר
                }
            }
        } else {
            // אם כמות הפריטים הכוללת לא תואמת בדיוק את סה"כ הנדרש להנחה, נבטל הנחות קודמות
            foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
                $cart_item['data']->set_price($cart_item['data']->get_regular_price());
            }
        }
    }
}
add_action('woocommerce_before_calculate_totals', 'bojo_apply_discount', 20);

// בדיקת מוצרים בעגלת הקניות והצגת פופ-אפ מותאם אישית
function bojo_check_cart_for_popup() {
    global $woocommerce;
    
    $cart_items = $woocommerce->cart->get_cart();
    $categories_to_check = get_option('bojo_categories_1'); // נבדוק רק קטגוריות בחלונית הראשונה
    $required_quantity = get_option('bojo_required_quantity_1');
    $free_quantity = get_option('bojo_free_quantity_1');
    
    if ($categories_to_check && $required_quantity && $free_quantity) {
        $categories = explode(',', $categories_to_check);
        $category_names = [];
        $total_qty = 0;

        foreach ($cart_items as $item) {
            $product_id = $item['product_id'];
            $terms = get_the_terms($product_id, 'product_cat');
            
            if ($terms) {
                foreach ($terms as $term) {
                    if (in_array($term->term_id, $categories)) {
                        $total_qty += $item['quantity'];
                        $category_names[] = $term->name;
                    }
                }
            }
        }

        $category_names = array_unique($category_names); // הסרת כפילויות
        $category_list = implode(', ', $category_names); // רשימת קטגוריות בפורמט טקסט

        // אם הכמות הכוללת היא בדיוק הכמות הנדרשת שהוגדרה ולא 13
        if ($total_qty == $required_quantity && $total_qty != 13) {
            echo '
            <script type="text/javascript">
                jQuery(document).ready(function($) {
                    setTimeout(function() {
                        var popup = document.getElementById("bojo-popup");
                        popup.style.display = "block";
                    }, 2000);
                });
            </script>';
        } elseif ($total_qty == 10) { 
            // הפופאפ יופיע רק כאשר הכמות הכוללת היא בדיוק 10
            echo '
            <script type="text/javascript">
                jQuery(document).ready(function($) {
                    setTimeout(function() {
                        var popup = document.getElementById("bojo-popup");
                        popup.style.display = "block";
                    }, 2000);
                });
            </script>';
        }
    }
}
add_action('wp_footer', 'bojo_check_cart_for_popup');
